// $Id: CTextEdit.cpp,v 1.9 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CTextEdit.hpp"
using Exponent::GUI::Controls::CTextEdit;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CTextEdit, CTextLabel);

#ifdef WIN32
//	===========================================================================
WNDPROC CTextEdit::m_lastWinProc = NULL;
#endif

#include <Host/CApplication.hpp>
using Exponent::Host::CApplication;

//	===========================================================================
CTextEdit::CTextEdit(IControlRoot *root, const long uniqueId, const CRect &area, const CString &text, IActionListener *listener) 
		 : CTextLabel(root, uniqueId, area, text)
		 , m_singleClickEdit(true)
		 #ifdef WIN32
		 , m_editWindowHandle(NULL)
		 , m_previouslyLockedControl(NULL)
		 #else
		 , m_txnHandle(NULL)
		 , m_textEditEventHandler(NULL)
		 , m_cancel(false)
		 #endif
		 , m_alignment(e_left)
		 , m_isEditable(true)
		 , m_limitText(true)
		 , m_textLimit(50)
{
	EXPONENT_CLASS_CONSTRUCTION(CTextEdit);

	this->controlIsMouseEnabled();
	this->registerActionListener(listener);
	this->controlIsDropFileEnabled(true);
}

//	===========================================================================
CTextEdit::~CTextEdit()
{
	EXPONENT_CLASS_DESTRUCTION(CTextEdit);
#ifdef WIN32
	if (m_editWindowHandle)
	{
		DestroyWindow(m_editWindowHandle);
	}
	NULL_POINTER(m_editWindowHandle);
	FORGET_COUNTED_OBJECT(m_previouslyLockedControl);
#else

#endif
}

//	===========================================================================
void CTextEdit::handleLeftButtonDown(CMouseEvent &event)
{
	if (!m_singleClickEdit)
	{
		return;
	}

	// Launch the edit
	this->launchEdit(event);
}

//	===========================================================================
void CTextEdit::handleDoubleClick(CMouseEvent &event)
{
	if (!m_isEditable || m_singleClickEdit)
	{
		return;
	}

	// Launch the edit
	this->launchEdit(event);
}

//	===========================================================================
void CTextEdit::launchEdit(CMouseEvent &event)
{
#ifdef WIN32

	// We stel the control locking
	EXCHANGE_COUNTED_OBJECTS(m_previouslyLockedControl, m_rootControl->getParentWindow()->getControlRoot()->getLockedControl());
	m_rootControl->getParentWindow()->getControlRoot()->lockControl(this);

	// Compute the area to place the control
	CRect clientArea;
	CWindowTools::getGlobalContentAreaCoordinates(m_rootControl->getParentWindow()->getMutableWindowHandle(), clientArea);
	
	CRect offsetArea = m_absoluteArea;
	offsetArea.offset(clientArea.getOrigin());
	//offsetArea.inset(2);

	// Check if we need to destroy the old window
	if (m_editWindowHandle)
	{
		DestroyWindow(m_editWindowHandle);
		NULL_POINTER(m_editWindowHandle);
	}

	// Create the edit window
	m_editWindowHandle = CreateWindow("EDIT",
									  m_text.getString(),
									  m_alignment | WS_POPUP | WS_VISIBLE | ES_AUTOHSCROLL,
									  offsetArea.getLeft(), offsetArea.getTop(), offsetArea.getWidth(), offsetArea.getHeight(),
									  m_rootControl->getParentWindow()->getWindowHandle()->m_windowHandle,
									  NULL,
									  CApplication::getApplicationInstance(),
									  NULL);

	// Store a pointer to the window
	SetWindowLongPtr(m_editWindowHandle, GWL_USERDATA, (long)this);

	// Set the font
	SendMessage(m_editWindowHandle, WM_SETFONT, (WPARAM)m_font->getFont(), true);

	// Set some options
	SendMessage(m_editWindowHandle, EM_SETMARGINS, EC_LEFTMARGIN | EC_RIGHTMARGIN, MAKELONG(0, 0));

	// Set the text selected
	SendMessage(m_editWindowHandle, EM_SETSEL, 0, -1);

	// If they want us to limit text we need to enable this
	if (m_limitText)
	{
		SendMessage(m_editWindowHandle, EM_LIMITTEXT, m_textLimit, 0);
	}

	// Ensure that the edit window has focus
	SetFocus(m_editWindowHandle);

	// Store the parent window proc
	CTextEdit::m_lastWinProc = (WNDPROC)SetWindowLongPtr(m_editWindowHandle, GWL_WNDPROC, (long)windowProcTextEdit);
#else
	// A few variables to play with
	WindowRef window			    = m_rootControl->getParentWindow()->getWindowHandle()->m_windowHandle;
	TXNFrameOptions txnFrameOptions = kTXNMonostyledTextMask | kTXNDisableDragAndDropMask | kTXNSingleLineOnlyMask;
	TXNObject txnObject				= 0;
	TXNFrameID frameID				= 0;
	TXNObjectRefcon txnObjectRefCon	= 0;
	
	//CRect clientArea;
	//CWindowTools::getGlobalContentAreaCoordinates(m_rootControl->getParentWindow()->getMutableWindowHandle(), clientArea);
	CRect offsetArea = m_absoluteArea;
	
	// The rect we are going to draw in to
	Rect *rect = offsetArea.getAsRect();

	if (TXNNewObject(NULL, 
					 window, 
					 rect, 
					 txnFrameOptions, 
					 kTXNTextEditStyleFrameType, 
					 kTXNSingleStylePerTextDocumentResType, 
					 kTXNMacOSEncoding, 
					 &txnObject, 
					 &frameID, 
					 txnObjectRefCon) == noErr)
	{
		TXNSetFrameBounds(txnObject, rect->top, rect->left, rect->bottom, rect->right, frameID);
		m_txnHandle = txnObject;
		
		// Set the text to display by defualt
		TXNSetData (m_txnHandle, kTXNTextData, m_text.getString(), strlen(m_text.getString()), kTXNStartOffset, kTXNEndOffset);
		
		// Set the background colours
		RGBColor *backColour   = m_backgroundColour.getAsRGBColour();
		RGBColor *textColour   = CColour::CCOLOUR_BLACK.getAsRGBColour();
		//RGBBackColor(backColour);
		//RGBForeColor(textColour);
		
		TXNBackground txnBackground;
		txnBackground.bgType         = kTXNBackgroundTypeRGB;
		txnBackground.bg.color.red   = backColour->red;
		txnBackground.bg.color.green = backColour->green;
		txnBackground.bg.color.blue  = backColour->blue;
		TXNSetBackground(m_txnHandle, &txnBackground);
		
		// Set justification
		SInt16 justification;
		Fract flushness;
		switch (m_alignment)
		{
			case e_left:	
				justification = kTXNFlushLeft;	
				flushness	  = kATSUStartAlignment;
			break;
			case e_right:	
				justification = kTXNFlushRight;	
				flushness	  = kATSUEndAlignment;
			break;
			case e_center:	
				justification = kTXNCenter;		
				flushness	  = kATSUCenterAlignment;
			break;
		}
		
		TXNControlTag	controlTag[1];
		TXNControlData	controlData[1];
		controlTag[0]		  = kTXNJustificationTag;
		controlData[0].sValue = justification;
		TXNSetTXNObjectControls (m_txnHandle, false, 1, controlTag, controlData);
		
		// Set the font
		short familyId;
		ATSUFontID fontNameId;
		
		CPascalString string(m_font->getFontName().getString());
		GetFNum(string.getUnsignedConstString(), &familyId);
		ATSUFONDtoFontID(familyId, 0, &fontNameId);
		
		// font
		TXNTypeAttributes attributes[3];
		attributes[0].tag          = kATSUFontTag;
		attributes[0].size         = sizeof(ATSUFontID);
		attributes[0].data.dataPtr = &fontNameId;
		// size
		attributes[1].tag			 = kTXNQDFontSizeAttribute;
		attributes[1].size			 = kTXNFontSizeAttributeSize;
		attributes[1].data.dataValue = m_font->getHeight() << 16;
		// color
		attributes[2].tag		   = kTXNQDFontColorAttribute;
		attributes[2].size		   = kTXNQDFontColorAttributeSize;
		attributes[2].data.dataPtr = textColour;
		
		// Finally set the attributes
		TXNSetTypeAttributes(m_txnHandle, 3, attributes, kTXNStartOffset, kTXNEndOffset);
		
		// Ensure focus remains consistent
		SetUserFocusWindow(window);
		AdvanceKeyboardFocus(window);
		
		// Set the focus to the edit window
		TXNFocus(txnObject, true);
		TXNSelectAll(m_txnHandle);
		TXNShowSelection(m_txnHandle, true);
		
		// The event types
		const static EventTypeSpec eventTypes[] = 
		{ 
			{ kEventClassMouse,    kEventMouseMoved		       }, 
			{ kEventClassMouse,    kEventMouseDown		       }, 
			{ kEventClassMouse,    kEventMouseUp		       }, 
			{ kEventClassWindow,   kEventWindowClosed          },
			{ kEventClassWindow,   kEventWindowDeactivated     }, 
			{ kEventClassWindow,   kEventWindowFocusRelinquish },
			{ kEventClassKeyboard, kEventRawKeyDown		       }, 
			{ kEventClassKeyboard, kEventRawKeyRepeat	       }
		};
		
		// Install the event handler
		InstallWindowEventHandler(window, windowProcTextEdit, GetEventTypeCount(eventTypes), eventTypes, this, &m_textEditEventHandler);
		
		delete backColour;
		delete textColour;
	}
#endif
}

//	===========================================================================
void CTextEdit::handleActionEvent(const CActionEvent &event)
{
#ifdef WIN32
	if (m_editWindowHandle)
	{
		// Create a buffer to hold the text copied..
		char *newText = new char[m_textLimit];

		// Get the actual text
		GetWindowText(m_editWindowHandle, newText, m_textLimit);

		// Check that we dont have a duff string
		if (strcmp(newText, "") == 0)
		{
			m_text.setString(" ");
		}
		else
		{
			m_text.setString(newText);
		}

		// We've copied it and can now delete the text
		FREE_ARRAY_POINTER(newText);

		// Finally we need to destroy the widnow
		DestroyWindow(m_editWindowHandle);
		
		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(event);
		}

		m_rootControl->getParentWindow()->getControlRoot()->unlockControl();
		m_rootControl->getParentWindow()->getControlRoot()->lockControl(m_previouslyLockedControl);
		FORGET_COUNTED_OBJECT(m_previouslyLockedControl);
		m_brush.uninitialise();
	}
	this->update();
#else
	if (!m_cancel)
	{
		// Get the text
		CharsHandle textHandle;
		TXNGetDataEncoded(m_txnHandle, kTXNStartOffset, kTXNEndOffset, &textHandle, kTXNTextData);
		
		// Check that we have some worthwhile data
		if (textHandle != NULL && GetHandleSize(textHandle) > 0)
		{
			const long textLength = GetHandleSize(textHandle);
			char text[257];
			strncpy(text, *textHandle, (textLength > 255) ? 255 : textLength);
			text[(textLength > 255) ? 255 : textLength] = '\0';
			m_text.setString(text);
		}
		
		// Kill the event handler
		if (m_textEditEventHandler) 
		{
			RemoveEventHandler(m_textEditEventHandler);
			NULL_POINTER(m_textEditEventHandler);
		}
		
		// Kill the edit box
		TXNFocus(m_txnHandle, false);
		TXNClear(m_txnHandle);
		TXNDeleteObject(m_txnHandle);
		NULL_POINTER(m_txnHandle);
			
		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(event);
		}
		this->update();
	}
	else
	{
		// Kill the event handler
		if (m_textEditEventHandler) 
		{
			RemoveEventHandler(m_textEditEventHandler);
			NULL_POINTER(m_textEditEventHandler);
		}
		
		// Kill the edit box
		if (m_txnHandle)
		{
			TXNFocus(m_txnHandle, false);
			TXNClear(m_txnHandle);
			TXNDeleteObject(m_txnHandle);
			NULL_POINTER(m_txnHandle);
		}
		this->update();
	}
	m_rootControl->getParentWindow()->redrawWindow();
#endif
}

#ifdef WIN32
//	===========================================================================
LRESULT CTextEdit::setEditWindowDrawConditions(HDC drawContext)
{
	SetBkMode(drawContext, TRANSPARENT);
	SetTextColor(drawContext, RGB(m_textColour.getRed(), m_textColour.getGreen(), m_textColour.getBlue()));
	m_brush.uninitialise();
	m_brush.initialise(drawContext, m_backgroundColour);
	return (LRESULT)m_brush.getMutableBrushHdc();
}

//	===========================================================================
LONG WINAPI CTextEdit::windowProcTextEdit(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_GETDLGCODE:
			{
				long flags = DLGC_WANTALLKEYS;
				return flags;
			}
		break;
		case WM_KEYDOWN:
			{
				if (wParam == VK_RETURN)
				{
					CTextEdit *textEdit = (CTextEdit*)GetWindowLong(windowHandle, GWL_USERDATA);
					if (textEdit)
					{
						textEdit->handleActionEvent(CActionEvent(textEdit));
					}
				}
			}
		break;
		case WM_KILLFOCUS:
		{
			CTextEdit *textEdit = (CTextEdit*)GetWindowLong(windowHandle, GWL_USERDATA);
			if (textEdit)
			{
				textEdit->handleActionEvent(CActionEvent(textEdit));
			}
		}
		break;
	}

	return (LONG)CallWindowProc(CTextEdit::m_lastWinProc, windowHandle, message, wParam, lParam);
}
#else
//	===========================================================================
pascal OSStatus CTextEdit::windowProcTextEdit(EventHandlerCallRef inHandlerCallRef, EventRef inEvent, void *inUserData)
{
	// The result we pass back
	OSStatus result = eventNotHandledErr;
	
	// The event class and kind
	EventClass eventClass = GetEventClass (inEvent);
	EventKind eventKind   = GetEventKind (inEvent);
	
	// The edit box that we are dealing with
	CTextEdit* textEdit = (CTextEdit*)inUserData;

	switch (eventClass)
	{
		case kEventClassKeyboard:
			switch (eventKind)
			{
				case kEventRawKeyDown:
				case kEventRawKeyRepeat:
					{
						// Get the keys and modifiers
						char macCharCode;
						UInt32 keyCode;
						UInt32 modifiers;
						GetEventParameter(inEvent, kEventParamKeyMacCharCodes, typeChar,   NULL, sizeof(char),   NULL, &macCharCode);
						GetEventParameter(inEvent, kEventParamKeyCode,         typeUInt32, NULL, sizeof(UInt32), NULL, &keyCode);
						GetEventParameter(inEvent, kEventParamKeyModifiers,    typeUInt32, NULL, sizeof(UInt32), NULL, &modifiers);
						
						// Check what the controls are
						if (macCharCode == 118 && modifiers == 256) 
						{
							// They have held down mac-v to paste a value in... Check its valid
							if (TXNIsScrapPastable()) 
							{
								TXNPaste(textEdit->getTxnHandle());
							}
						}
						else if (macCharCode == 13 || macCharCode == 3 || macCharCode == 27)
						{
							if (macCharCode == 27)
							{
								//CTrace::trace("Cancel");
								textEdit->cancelCall(true);
							}
							else
							{
								textEdit->cancelCall(false);
							}
							textEdit->handleActionEvent(CActionEvent(textEdit));
							result = noErr;
						}
						else
						{
							EventRecord eventRecord;
							if (ConvertEventRefToEventRecord(inEvent, &eventRecord))
							{
								TXNKeyDown(textEdit->getTxnHandle(), &eventRecord);
							}
							result = noErr;
						}
					}
				break;
			}
		break;
		case kEventClassMouse:
		{
			switch (eventKind)
			{
				case kEventMouseDown:
				case kEventMouseUp:
					{
						// Get the window handle
						WindowRef window;
						GetEventParameter(inEvent, kEventParamWindowRef, typeWindowRef, NULL, sizeof(WindowRef), NULL, &window);
						
						// Determine the point
						HIPoint p;
						GetEventParameter(inEvent, kEventParamMouseLocation, typeHIPoint, NULL, sizeof(HIPoint), NULL, &p);
						Point point = { (short)p.y, (short)p.x };
						QDGlobalToLocalPoint(GetWindowPort (window), &point);
						
						// Get the viewable area
						Rect rect;
						TXNGetViewRect (textEdit->getTxnHandle(), &rect);
						
						// Handle the click as necessary
						if (PtInRect(point, &rect))
						{
							EventRecord eventRecord;
							if (eventKind == kEventMouseDown && ConvertEventRefToEventRecord(inEvent, &eventRecord))
							{
								TXNClick(textEdit->getTxnHandle(), &eventRecord);
							}
							result = noErr;
						}
						else
						{
							result = CallNextEventHandler(inHandlerCallRef, inEvent);
							ClearKeyboardFocus(window);
							textEdit->cancelCall(true);
							textEdit->handleActionEvent(CActionEvent(textEdit));
							result = noErr;
						}
					}
				break;
				case kEventMouseMoved:
					TXNAdjustCursor(textEdit->getTxnHandle(), NULL);
				break;
			}
			break;
		}
		case kEventClassWindow:
		{
			WindowRef window;
			GetEventParameter (inEvent, kEventParamDirectObject, typeWindowRef, NULL, sizeof(WindowRef), NULL, &window);
			switch (eventKind)
			{
				case kEventWindowFocusRelinquish:
				case kEventWindowClosed:
				case kEventWindowDeactivated:
					result = CallNextEventHandler(inHandlerCallRef, inEvent);
					ClearKeyboardFocus(window);
					textEdit->cancelCall(true);
					textEdit->handleActionEvent(CActionEvent(textEdit));
					result = noErr;
				break;
			}
			break;
		}
	}
	return result;
}

//	===========================================================================
void CTextEdit::drawControl(CGraphics &graphics)
{
	if (m_txnHandle)
	{
		TXNDraw(m_txnHandle, NULL);
		return;
	}
	CTextLabel::drawControl(graphics);
}
#endif

//	===========================================================================
void CTextEdit::limitText(const bool limitText, const long textLimit)
{
	m_limitText = limitText;
	m_textLimit = textLimit;
}